# Guía de Instalación - Request Validation Helpers Block

Esta guía te ayudará a integrar el bloque de Request Validation Helpers en tu proyecto Laravel 12.

## 📋 Requisitos Previos

- Laravel 12.x instalado
- PHP 8.3 o superior

## 🚀 Pasos de Instalación

### Paso 1: Extraer el contenido

**Importante:** Descomprime el archivo ZIP en la **raíz de tu proyecto Laravel** (donde están las carpetas `app/`, `config/`, `routes/`, `bootstrap/`, etc.).

Al descomprimir el ZIP, la estructura de archivos quedará así:

```
tu-proyecto-laravel/
├── app/
│   └── Blocks/
│       └── Validation/              ← El bloque queda aquí
│           ├── Requests/
│           ├── Rules/
│           └── ValidationServiceProvider.php
├── README.md
├── LICENSE.txt
└── docs/
    └── INSTALLATION.md
```

**Nota:** Asegúrate de descomprimir el ZIP en la raíz del proyecto, no dentro de una subcarpeta. Los archivos deben quedar directamente en las carpetas `app/`, etc.

### Paso 2: Usar los Form Requests

Este bloque NO requiere registrar ningún Service Provider. Los Form Requests de Laravel se cargan automáticamente cuando los usas en tus controladores.

Simplemente importa y usa los Form Requests en tus controladores:

```php
use App\Blocks\Validation\Requests\EmailRequest;
use App\Blocks\Validation\Requests\PasswordRequest;
use App\Blocks\Validation\Requests\UserDataRequest;
use App\Blocks\Validation\Requests\PaginationRequest;
use App\Blocks\Validation\Requests\SearchRequest;
```

## ✅ Verificación

Para verificar que todo funciona correctamente:

1. **Importa un Form Request en un controlador:**
   ```php
   use App\Blocks\Validation\Requests\EmailRequest;
   ```

2. **Úsalo en un método:**
   ```php
   public function store(EmailRequest $request)
   {
       $email = $request->validated('email');
       // El email está validado
   }
   ```

3. **Prueba la validación:**
   - Envía una petición con un email inválido
   - Deberías recibir mensajes de error apropiados

## 🔧 Ejemplos de Uso

### Validar Email

```php
use App\Blocks\Validation\Requests\EmailRequest;
use Illuminate\Http\RedirectResponse;

public function sendEmail(EmailRequest $request): RedirectResponse
{
    $email = $request->validated('email');
    
    // Procesar el email
    Mail::to($email)->send(new WelcomeMail());
    
    return redirect()->back()->with('success', 'Email enviado.');
}
```

### Validar Contraseña

```php
use App\Blocks\Validation\Requests\PasswordRequest;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Hash;

public function updatePassword(PasswordRequest $request): RedirectResponse
{
    $password = $request->validated('password');
    
    auth()->user()->update([
        'password' => Hash::make($password),
    ]);
    
    return redirect()->back()->with('success', 'Contraseña actualizada.');
}
```

### Validar Datos de Usuario

```php
use App\Blocks\Validation\Requests\UserDataRequest;
use App\Models\User;
use Illuminate\Http\RedirectResponse;

public function createUser(UserDataRequest $request): RedirectResponse
{
    $data = $request->validated();
    
    User::create($data);
    
    return redirect()->route('users.index')->with('success', 'Usuario creado.');
}
```

### Paginación

```php
use App\Blocks\Validation\Requests\PaginationRequest;
use App\Models\User;
use Illuminate\View\View;

public function index(PaginationRequest $request): View
{
    $params = $request->getPaginationParams();
    
    $users = User::paginate(
        $params['per_page'],
        ['*'],
        'page',
        $params['page']
    );
    
    return view('users.index', compact('users'));
}
```

### Búsqueda

```php
use App\Blocks\Validation\Requests\SearchRequest;
use App\Models\User;
use Illuminate\View\View;

public function search(SearchRequest $request): View
{
    $searchTerm = $request->getSearchTerm();
    
    $query = User::query();
    
    if ($searchTerm) {
        $query->where('name', 'like', "%{$searchTerm}%");
    }
    
    $users = $query->get();
    
    return view('users.search', compact('users'));
}
```

## 🔧 Solución de Problemas

### Error: "Class not found"

1. **Verifica que los archivos estén en la ubicación correcta:**
   ```bash
   ls -la app/Blocks/Validation/Requests/
   ```

2. **Ejecuta composer dump-autoload:**
   ```bash
   composer dump-autoload
   ```

3. **Verifica los namespaces:**
   Asegúrate de que los namespaces en los archivos PHP coincidan con la estructura de directorios.

### La validación no funciona

1. **Verifica que estés usando el Form Request correctamente:**
   ```php
   // Correcto
   public function store(EmailRequest $request)
   
   // Incorrecto
   public function store(Request $request)
   ```

2. **Verifica que el campo en el formulario coincida con las reglas:**
   Si el Form Request valida `email`, el campo del formulario debe llamarse `email`.

### Los mensajes de error están en inglés

Los mensajes de error están configurados en español en los Form Requests. Si ves mensajes en inglés, es porque Laravel está usando mensajes por defecto. Asegúrate de que los métodos `messages()` en los Form Requests estén correctamente implementados.

## 📝 Personalización

### Extender un Form Request

Puedes extender los Form Requests para agregar validaciones específicas:

```php
use App\Blocks\Validation\Requests\UserDataRequest;

class CreateUserRequest extends UserDataRequest
{
    public function rules(): array
    {
        $rules = parent::rules();
        
        // Agregar regla de unicidad para email
        $rules['email'][] = 'unique:users,email';
        
        return $rules;
    }
    
    public function messages(): array
    {
        $messages = parent::messages();
        
        // Agregar mensaje personalizado
        $messages['email.unique'] = 'Este correo electrónico ya está registrado.';
        
        return $messages;
    }
}
```

### Crear Form Requests propios basados en los helpers

```php
use App\Blocks\Validation\Requests\EmailRequest;
use Illuminate\Validation\Rule;

class UpdateEmailRequest extends EmailRequest
{
    public function rules(): array
    {
        $rules = parent::rules();
        
        // Ignorar el email del usuario actual en la validación de unicidad
        $rules['email'][] = Rule::unique('users')->ignore(auth()->id());
        
        return $rules;
    }
}
```

## 📚 Documentación Adicional

Para más información, consulta:
- `README.md` - Documentación completa del bloque
- [Laravel Form Request Validation](https://laravel.com/docs/validation#form-request-validation) - Documentación oficial de Laravel

## 🎯 Próximos Pasos

Una vez instalado el bloque:

1. Revisa los Form Requests disponibles en `app/Blocks/Validation/Requests/`
2. Úsalos en tus controladores según tus necesidades
3. Extiéndelos cuando necesites validaciones específicas
4. ¡Disfruta de validaciones robustas y reutilizables!

---

**¿Necesitas ayuda?** Consulta la documentación de Laravel sobre [validación](https://laravel.com/docs/validation) y [Form Requests](https://laravel.com/docs/validation#form-request-validation).
