# Request Validation Helpers Block - Laravel 12

Un conjunto reutilizable de Form Requests y reglas de validación comunes para Laravel 12, diseñado para acelerar el desarrollo con validaciones robustas y bien estructuradas.

## 📋 Descripción

Este **Request Validation Helpers Block** proporciona un conjunto de Form Requests listos para usar en casos comunes de validación:

- Validación de email
- Validación de contraseña
- Validación de datos de usuario (nombre, email)
- Validación de parámetros de paginación
- Validación de búsquedas y filtros

**Importante:** Este es un bloque helper que proporciona Form Requests reutilizables. No incluye vistas, controladores, middleware ni lógica de negocio. Está diseñado para ser integrado en proyectos Laravel 12 donde necesites validaciones comunes.

## 🎯 Características

- ✅ Form Requests reutilizables para casos comunes
- ✅ Mensajes de error claros y en español
- ✅ Uso de reglas modernas de Laravel 12
- ✅ Custom Rule para contraseñas fuertes
- ✅ Métodos helper en Form Requests (paginación, búsqueda)
- ✅ Código limpio y bien estructurado
- ✅ Tipado estricto
- ✅ Sin dependencias externas

## 📦 Requisitos

- Laravel 12.x
- PHP 8.3 o superior

## 🚀 Instalación

### 1. Copiar archivos

Copia la carpeta `app/Blocks/Validation/` a tu proyecto Laravel en la misma ubicación.

### 2. Usar en tus controladores

Simplemente usa los Form Requests en tus controladores:

```php
use App\Blocks\Validation\Requests\EmailRequest;
use App\Blocks\Validation\Requests\PasswordRequest;
use App\Blocks\Validation\Requests\UserDataRequest;
use App\Blocks\Validation\Requests\PaginationRequest;
use App\Blocks\Validation\Requests\SearchRequest;
```

## 💡 Uso

### EmailRequest

Valida un campo de email con formato RFC y DNS:

```php
use App\Blocks\Validation\Requests\EmailRequest;

public function sendEmail(EmailRequest $request)
{
    $email = $request->validated('email');
    // El email ya está validado
}
```

### PasswordRequest

Valida una contraseña con confirmación y reglas de seguridad:

```php
use App\Blocks\Validation\Requests\PasswordRequest;

public function updatePassword(PasswordRequest $request)
{
    $password = $request->validated('password');
    // La contraseña ya está validada y confirmada
}
```

### UserDataRequest

Valida nombre y email para formularios de usuario:

```php
use App\Blocks\Validation\Requests\UserDataRequest;

public function createUser(UserDataRequest $request)
{
    $data = $request->validated();
    // $data['name'] y $data['email'] ya están validados
}
```

### PaginationRequest

Valida parámetros de paginación con valores por defecto:

```php
use App\Blocks\Validation\Requests\PaginationRequest;

public function index(PaginationRequest $request)
{
    $params = $request->getPaginationParams();
    // $params['page'] = 1 (default)
    // $params['per_page'] = 15 (default)
    
    $users = User::paginate($params['per_page'], ['*'], 'page', $params['page']);
}
```

### SearchRequest

Valida términos de búsqueda (soporta 'search' y 'query'):

```php
use App\Blocks\Validation\Requests\SearchRequest;

public function search(SearchRequest $request)
{
    $searchTerm = $request->getSearchTerm();
    // Retorna el término de búsqueda o null
    
    if ($searchTerm) {
        $results = User::where('name', 'like', "%{$searchTerm}%")->get();
    }
}
```

## 📁 Estructura del Bloque

```
validation-helpers-laravel12/
├── app/
│   └── Blocks/
│       └── Validation/
│           ├── Requests/
│           │   ├── EmailRequest.php
│           │   ├── PasswordRequest.php
│           │   ├── UserDataRequest.php
│           │   ├── PaginationRequest.php
│           │   └── SearchRequest.php
│           ├── Rules/
│           │   └── StrongPassword.php
│           └── ValidationServiceProvider.php
├── docs/
│   └── INSTALLATION.md
├── README.md
└── LICENSE.txt
```

## 🔧 Personalización

Puedes extender o modificar los Form Requests según tus necesidades:

```php
use App\Blocks\Validation\Requests\UserDataRequest;

class CreateUserRequest extends UserDataRequest
{
    public function rules(): array
    {
        $rules = parent::rules();
        
        // Agregar reglas adicionales
        $rules['email'][] = 'unique:users,email';
        
        return $rules;
    }
}
```

## 📝 Ejemplos de Uso en Controladores

### Ejemplo completo: Controlador de Usuarios

```php
use App\Blocks\Validation\Requests\UserDataRequest;
use App\Blocks\Validation\Requests\PasswordRequest;
use App\Blocks\Validation\Requests\PaginationRequest;
use App\Blocks\Validation\Requests\SearchRequest;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class UserController extends Controller
{
    public function index(PaginationRequest $pagination, SearchRequest $search): View
    {
        $params = $pagination->getPaginationParams();
        $searchTerm = $search->getSearchTerm();
        
        $query = User::query();
        
        if ($searchTerm) {
            $query->where('name', 'like', "%{$searchTerm}%");
        }
        
        $users = $query->paginate($params['per_page'], ['*'], 'page', $params['page']);
        
        return view('users.index', compact('users'));
    }
    
    public function store(UserDataRequest $userData, PasswordRequest $password): RedirectResponse
    {
        $data = array_merge(
            $userData->validated(),
            ['password' => bcrypt($password->validated('password'))]
        );
        
        User::create($data);
        
        return redirect()->route('users.index')->with('success', 'Usuario creado exitosamente.');
    }
}
```

## ⚠️ Notas Importantes

- Este bloque NO incluye vistas, controladores, middleware ni lógica de negocio
- Los Form Requests deben ser usados en tus controladores
- Puedes extender los Form Requests para agregar validaciones específicas
- Los mensajes de error están en español por defecto
- Las reglas de validación siguen las mejores prácticas de Laravel 12

## 🎯 Compatibilidad

- Laravel 12.x
- PHP 8.3+

## 📄 Licencia

Este bloque es gratuito y está disponible bajo la licencia MIT. Ver `LICENSE.txt` para más detalles.

---

**Request Validation Helpers Block v1.0.0** - Laravel 12
