# Guía de Instalación - Query Filters Block (Light)

Esta guía te ayudará a integrar el bloque de Query Filters en tu proyecto Laravel 12.

## 📋 Requisitos Previos

- Laravel 12.x instalado
- PHP 8.3 o superior

## 🚀 Pasos de Instalación

### Paso 1: Extraer el contenido

**Importante:** Descomprime el archivo ZIP en la **raíz de tu proyecto Laravel** (donde están las carpetas `app/`, `config/`, `routes/`, `bootstrap/`, etc.).

Al descomprimir el ZIP, la estructura de archivos quedará así:

```
tu-proyecto-laravel/
├── app/
│   └── Blocks/
│       └── QueryFilters/              ← El bloque queda aquí
│           ├── Filters/
│           │   ├── QueryFilter.php
│           │   └── ExampleFilters.php
│           ├── Traits/
│           │   └── Filterable.php
│           └── QueryFiltersServiceProvider.php
├── README.md
├── LICENSE.txt
└── docs/
    └── INSTALLATION.md
```

**Nota:** Asegúrate de descomprimir el ZIP en la raíz del proyecto, no dentro de una subcarpeta. Los archivos deben quedar directamente en las carpetas `app/`, etc.

### Paso 2: Agregar el trait a tu modelo

Agrega el trait `Filterable` a los modelos donde quieras usar filtros:

```php
use App\Blocks\QueryFilters\Traits\Filterable;
use Illuminate\Database\Eloquent\Model;

class User extends Model
{
    use Filterable;
    
    protected $fillable = ['name', 'email', 'status'];
}
```

### Paso 3: Crear filtros personalizados

Copia `ExampleFilters` y personalízalo para tu modelo. Por ejemplo, crea `UserFilters.php`:

```php
use App\Blocks\QueryFilters\Filters\QueryFilter;

class UserFilters extends QueryFilter
{
    public function search(string $value): void
    {
        $this->builder->where('name', 'like', "%{$value}%");
    }
    
    public function status(string $value): void
    {
        $this->builder->where('status', $value);
    }
}
```

### Paso 4: Usar en controladores

Usa los filtros en tus controladores:

```php
use App\Blocks\QueryFilters\Filters\UserFilters;
use App\Models\User;
use Illuminate\Http\Request;

class UserController extends Controller
{
    public function index(Request $request)
    {
        $filters = new UserFilters($request);
        $users = User::filter($filters)->get();
        
        return view('users.index', compact('users'));
    }
}
```

## ✅ Verificación

Para verificar que todo funciona correctamente:

1. **Agrega el trait a un modelo:**
   ```php
   use App\Blocks\QueryFilters\Traits\Filterable;
   
   class User extends Model
   {
       use Filterable;
   }
   ```

2. **Crea una clase de filtros:**
   ```php
   use App\Blocks\QueryFilters\Filters\QueryFilter;
   
   class UserFilters extends QueryFilter
   {
       public function search(string $value): void
       {
           $this->builder->where('name', 'like', "%{$value}%");
       }
   }
   ```

3. **Úsalo en un controlador:**
   ```php
   $filters = new UserFilters($request);
   $users = User::filter($filters)->get();
   ```

4. **Prueba con una URL:**
   ```
   /users?search=john
   ```

## 🔧 Ejemplos de Uso

### Ejemplo 1: Búsqueda simple

```php
// UserFilters.php
public function search(string $value): void
{
    $this->builder->where('name', 'like', "%{$value}%");
}

// Controlador
$filters = new UserFilters($request);
$users = User::filter($filters)->get();

// URL: /users?search=john
```

### Ejemplo 2: Múltiples filtros

```php
// UserFilters.php
public function search(string $value): void
{
    $this->builder->where('name', 'like', "%{$value}%");
}

public function status(string $value): void
{
    $this->builder->where('status', $value);
}

public function active(string $value): void
{
    $active = filter_var($value, FILTER_VALIDATE_BOOLEAN);
    $this->builder->where('is_active', $active);
}

// Controlador
$filters = new UserFilters($request);
$users = User::filter($filters)->paginate(15);

// URL: /users?search=john&status=active&active=1
```

### Ejemplo 3: Ordenamiento

```php
// UserFilters.php
public function order_by(string $column): void
{
    $direction = $this->request->get('order_direction', 'asc');
    $this->builder->orderBy($column, $direction);
}

// URL: /users?order_by=name&order_direction=desc
```

### Ejemplo 4: Rango de fechas

```php
// UserFilters.php
public function date_from(string $value): void
{
    $this->builder->whereDate('created_at', '>=', $value);
}

public function date_to(string $value): void
{
    $this->builder->whereDate('created_at', '<=', $value);
}

// URL: /users?date_from=2024-01-01&date_to=2024-12-31
```

### Ejemplo 5: Con paginación

```php
public function index(Request $request)
{
    $filters = new UserFilters($request);
    
    $users = User::filter($filters)
                ->paginate($request->get('per_page', 15));
    
    return view('users.index', compact('users'));
}
```

## 🔧 Solución de Problemas

### Error: "Call to undefined method filter()"

1. **Verifica que el trait esté agregado al modelo:**
   ```php
   use App\Blocks\QueryFilters\Traits\Filterable;
   
   class User extends Model
   {
       use Filterable; // ← Debe estar aquí
   }
   ```

2. **Ejecuta composer dump-autoload:**
   ```bash
   composer dump-autoload
   ```

### Los filtros no se aplican

1. **Verifica que el método en la clase de filtros tenga el mismo nombre que el parámetro de la URL:**
   ```php
   // Si la URL es: ?search=john
   // El método debe ser: search()
   public function search(string $value): void
   {
       // ...
   }
   ```

2. **Verifica que el valor no sea null ni string vacío:**
   Los filtros solo se aplican si el valor tiene contenido.

### Error: "Class not found"

1. **Verifica que los archivos estén en la ubicación correcta:**
   ```bash
   ls -la app/Blocks/QueryFilters/Filters/
   ls -la app/Blocks/QueryFilters/Traits/
   ```

2. **Ejecuta composer dump-autoload:**
   ```bash
   composer dump-autoload
   ```

3. **Verifica los namespaces:**
   Asegúrate de que los namespaces en los archivos PHP coincidan con la estructura de directorios.

## 📝 Personalización

### Crear filtros personalizados

Simplemente agrega métodos a tu clase de filtros. El nombre del método debe coincidir con el parámetro de la URL:

```php
class UserFilters extends QueryFilter
{
    // URL: /users?role=admin
    public function role(string $value): void
    {
        $this->builder->where('role', $value);
    }
    
    // URL: /users?min_age=18
    public function min_age(string $value): void
    {
        $this->builder->where('age', '>=', $value);
    }
}
```

### Combinar con scopes de Eloquent

```php
public function index(Request $request)
{
    $filters = new UserFilters($request);
    
    $users = User::active()  // Scope de Eloquent
                ->filter($filters)  // Filtros del bloque
                ->get();
    
    return view('users.index', compact('users'));
}
```

### Filtros más complejos

```php
public function price_range(string $value): void
{
    $range = explode('-', $value);
    if (count($range) === 2) {
        $this->builder->whereBetween('price', [
            (int) $range[0],
            (int) $range[1]
        ]);
    }
}

// URL: /products?price_range=100-500
```

## 📚 Documentación Adicional

Para más información, consulta:
- `README.md` - Documentación completa del bloque
- [Laravel Eloquent](https://laravel.com/docs/eloquent) - Documentación oficial de Laravel

## 🎯 Próximos Pasos

Una vez instalado el bloque:

1. Agrega el trait `Filterable` a tus modelos
2. Crea clases de filtros personalizadas para cada modelo
3. Úsalas en tus controladores
4. ¡Disfruta de filtros reutilizables y mantenibles!

---

**¿Necesitas ayuda?** Consulta la documentación de Laravel sobre [Eloquent](https://laravel.com/docs/eloquent) y [Query Builder](https://laravel.com/docs/queries).
