# Query Filters Block (Light) - Laravel 12

Un bloque de filtros de consulta elegante y reutilizable para Laravel 12, diseñado para aplicar filtros a consultas Eloquent de forma clara y mantenible usando clases dedicadas.

## 📋 Descripción

Este **Query Filters Block (Light)** proporciona una forma elegante y reutilizable de aplicar filtros a consultas Eloquent usando clases dedicadas y buenas prácticas:

- Clase base abstracta `QueryFilter` para crear filtros personalizados
- Trait `Filterable` para aplicar filtros desde modelos
- Ejemplo completo con filtros comunes (búsqueda, ordenamiento, estado, fechas)
- Código limpio y fácil de entender

**Importante:** Este es un bloque LIGHT que proporciona una estructura base para filtros. No incluye vistas, controladores, middleware ni lógica de negocio compleja. Está diseñado para ser integrado en proyectos Laravel 12 donde necesites filtros reutilizables.

## 🎯 Características

- ✅ Clase base abstracta `QueryFilter` reutilizable
- ✅ Trait `Filterable` para usar en modelos
- ✅ Ejemplo completo con filtros comunes
- ✅ Filtros incluidos: búsqueda, ordenamiento, estado, rango de fechas
- ✅ Código limpio y bien estructurado
- ✅ Tipado estricto
- ✅ Sin dependencias externas
- ✅ Fácil de extender y personalizar

## 📦 Requisitos

- Laravel 12.x
- PHP 8.3 o superior

## 🚀 Instalación

### 1. Copiar archivos

Copia la carpeta `app/Blocks/QueryFilters/` a tu proyecto Laravel en la misma ubicación.

### 2. Usar el trait en tus modelos

Agrega el trait `Filterable` a los modelos donde quieras usar filtros:

```php
use App\Blocks\QueryFilters\Traits\Filterable;
use Illuminate\Database\Eloquent\Model;

class User extends Model
{
    use Filterable;
    
    // ...
}
```

### 3. Crear filtros personalizados

Copia `ExampleFilters` y personalízalo para tu modelo:

```php
use App\Blocks\QueryFilters\Filters\QueryFilter;

class UserFilters extends QueryFilter
{
    public function search(string $value): void
    {
        $this->builder->where('name', 'like', "%{$value}%");
    }
    
    // Agrega más métodos de filtro...
}
```

## 💡 Uso

### Uso básico en controladores

```php
use App\Blocks\QueryFilters\Filters\ExampleFilters;
use App\Models\User;
use Illuminate\Http\Request;

class UserController extends Controller
{
    public function index(Request $request)
    {
        $filters = new ExampleFilters($request);
        $users = User::filter($filters)->get();
        
        return view('users.index', compact('users'));
    }
}
```

### Con paginación

```php
public function index(Request $request)
{
    $filters = new ExampleFilters($request);
    $users = User::filter($filters)->paginate(15);
    
    return view('users.index', compact('users'));
}
```

### Con relaciones

```php
public function index(Request $request)
{
    $filters = new ExampleFilters($request);
    $users = User::with('profile')
                 ->filter($filters)
                 ->get();
    
    return view('users.index', compact('users'));
}
```

## 📁 Estructura del Bloque

```
query-filters-laravel12/
├── app/
│   └── Blocks/
│       └── QueryFilters/
│           ├── Filters/
│           │   ├── QueryFilter.php (clase base)
│           │   └── ExampleFilters.php (ejemplo)
│           ├── Traits/
│           │   └── Filterable.php
│           └── QueryFiltersServiceProvider.php
├── docs/
│   └── INSTALLATION.md
├── README.md
└── LICENSE.txt
```

## 🔧 Filtros Incluidos en el Ejemplo

### Búsqueda (`search`)

Busca en múltiples campos:

```php
// URL: /users?search=john
public function search(string $value): void
{
    $this->builder->where(function ($query) use ($value) {
        $query->where('name', 'like', "%{$value}%")
              ->orWhere('email', 'like', "%{$value}%");
    });
}
```

### Ordenamiento (`order_by` y `order_direction`)

Ordena por columna y dirección:

```php
// URL: /users?order_by=name&order_direction=asc
public function order_by(string $column): void
{
    $direction = $this->request->get('order_direction', 'asc');
    $this->builder->orderBy($column, $direction);
}
```

### Estado (`status`)

Filtra por estado:

```php
// URL: /users?status=active
public function status(string $value): void
{
    $this->builder->where('status', $value);
}
```

### Rango de fechas (`date_from` y `date_to`)

Filtra por rango de fechas:

```php
// URL: /users?date_from=2024-01-01&date_to=2024-12-31
public function date_from(string $value): void
{
    $this->builder->whereDate('created_at', '>=', $value);
}

public function date_to(string $value): void
{
    $this->builder->whereDate('created_at', '<=', $value);
}
```

### Estado activo (`active`)

Filtra por estado booleano:

```php
// URL: /users?active=1
public function active(string $value): void
{
    $active = filter_var($value, FILTER_VALIDATE_BOOLEAN);
    $this->builder->where('is_active', $active);
}
```

## 📝 Ejemplo Completo

### Modelo

```php
use App\Blocks\QueryFilters\Traits\Filterable;
use Illuminate\Database\Eloquent\Model;

class User extends Model
{
    use Filterable;
    
    protected $fillable = ['name', 'email', 'status', 'is_active'];
}
```

### Filtros personalizados

```php
use App\Blocks\QueryFilters\Filters\QueryFilter;

class UserFilters extends QueryFilter
{
    public function search(string $value): void
    {
        $this->builder->where('name', 'like', "%{$value}%");
    }
    
    public function status(string $value): void
    {
        $this->builder->where('status', $value);
    }
}
```

### Controlador

```php
use App\Blocks\QueryFilters\Filters\UserFilters;
use App\Models\User;
use Illuminate\Http\Request;

class UserController extends Controller
{
    public function index(Request $request)
    {
        $filters = new UserFilters($request);
        
        $users = User::filter($filters)
                    ->paginate(15);
        
        return view('users.index', compact('users'));
    }
}
```

### Uso en la URL

```
/users?search=john&status=active&order_by=name&order_direction=desc
```

## 🔧 Personalización

### Crear nuevos filtros

Simplemente agrega métodos a tu clase de filtros:

```php
class UserFilters extends QueryFilter
{
    public function role(string $value): void
    {
        $this->builder->where('role', $value);
    }
    
    public function created_after(string $value): void
    {
        $this->builder->where('created_at', '>=', $value);
    }
}
```

### Combinar múltiples condiciones

```php
public function search(string $value): void
{
    $this->builder->where(function ($query) use ($value) {
        $query->where('name', 'like', "%{$value}%")
              ->orWhere('email', 'like', "%{$value}%")
              ->orWhere('phone', 'like', "%{$value}%");
    });
}
```

### Filtros más complejos

```php
public function price_range(string $value): void
{
    $range = explode('-', $value);
    if (count($range) === 2) {
        $this->builder->whereBetween('price', [$range[0], $range[1]]);
    }
}
```

## ⚠️ Notas Importantes

- Este bloque NO incluye vistas, controladores, middleware ni lógica de negocio
- Los filtros deben ser creados para cada modelo según sus necesidades
- El trait `Filterable` debe ser agregado a cada modelo que use filtros
- Los filtros solo se aplican si el valor no es null ni string vacío
- Puedes combinar múltiples filtros en una misma consulta

## 🎯 Compatibilidad

- Laravel 12.x
- PHP 8.3+

## 📄 Licencia

Este bloque es gratuito y está disponible bajo la licencia MIT. Ver `LICENSE.txt` para más detalles.

---

**Query Filters Block (Light) v1.0.0** - Laravel 12
