# Guía de Instalación - Model Cleanup & Maintenance Block (Light)

Esta guía te ayudará a integrar el bloque Model Cleanup & Maintenance en tu proyecto Laravel 12.

## 📋 Requisitos Previos

- Laravel 12.x instalado
- PHP 8.3 o superior

## 🚀 Pasos de Instalación

### Paso 1: Extraer el contenido

**Importante:** Descomprime el archivo ZIP en la **raíz de tu proyecto Laravel** (donde están las carpetas `app/`, `config/`, `routes/`, `bootstrap/`, etc.).

Al descomprimir el ZIP, la estructura de archivos quedará así:

```
tu-proyecto-laravel/
├── app/
│   └── Blocks/
│       └── Maintenance/              ← El bloque queda aquí
│           ├── Commands/
│           ├── Services/
│           └── MaintenanceServiceProvider.php
├── config/
│   └── maintenance.php               ← Archivo de configuración
├── README.md
├── LICENSE.txt
└── docs/
    └── INSTALLATION.md
```

**Nota:** Asegúrate de descomprimir el ZIP en la raíz del proyecto, no dentro de una subcarpeta. Los archivos deben quedar directamente en las carpetas `app/`, `config/`, etc.

### Paso 2: Registrar el Service Provider

Abre el archivo `bootstrap/providers.php` y añade el Service Provider del bloque:

```php
<?php

return [
    App\Providers\AppServiceProvider::class,
    App\Blocks\Maintenance\MaintenanceServiceProvider::class, // ← Añadir esta línea
];
```

### Paso 3: Verificar configuración

El archivo `config/maintenance.php` se copiará automáticamente. Puedes personalizarlo si es necesario.

### Paso 4: Limpiar caché (opcional)

```bash
php artisan config:clear
```

## ✅ Verificación

Para verificar que todo funciona correctamente:

1. **Verifica que los comandos estén registrados:**
   ```bash
   php artisan list | grep maintenance
   ```
   
   Deberías ver:
   - `maintenance:clean-orphaned`
   - `maintenance:clean-soft-deleted`
   - `maintenance:clean-old-records`

2. **Prueba un comando (sin ejecutar):**
   ```bash
   php artisan maintenance:clean-soft-deleted "App\Models\User" --days=30
   ```
   
   Deberías ver un mensaje de confirmación.

## 🔧 Ejemplos de Uso

### Limpiar soft-deleted records

```bash
# Con confirmación (default)
php artisan maintenance:clean-soft-deleted "App\Models\User" --days=30

# Sin confirmación (--force)
php artisan maintenance:clean-soft-deleted "App\Models\User" --days=30 --force
```

### Limpiar registros huérfanos

```bash
# Limpiar comentarios sin usuario válido
php artisan maintenance:clean-orphaned "App\Models\Comment" user

# Con tabla padre especificada
php artisan maintenance:clean-orphaned "App\Models\Comment" user --parent-table=users
```

### Limpiar registros antiguos

```bash
# Limpiar logs más antiguos que 90 días
php artisan maintenance:clean-old-records logs created_at --days=90

# Limpiar sesiones antiguas
php artisan maintenance:clean-old-records sessions last_activity --days=30
```

## 🔧 Solución de Problemas

### Error: "Class does not exist"

1. **Verifica que el modelo esté correctamente referenciado:**
   ```bash
   # Correcto
   php artisan maintenance:clean-soft-deleted "App\Models\User"
   
   # Incorrecto (sin namespace completo)
   php artisan maintenance:clean-soft-deleted "User"
   ```

2. **Ejecuta composer dump-autoload:**
   ```bash
   composer dump-autoload
   ```

### Error: "must use SoftDeletes trait"

El modelo debe usar el trait `SoftDeletes` de Laravel:

```php
use Illuminate\Database\Eloquent\SoftDeletes;

class User extends Model
{
    use SoftDeletes; // ← Debe estar presente
}
```

### Error: "Table does not exist"

Verifica que el nombre de la tabla sea correcto:

```bash
php artisan tinker
>>> DB::table('nombre_tabla')->count();
```

### Los comandos no aparecen

1. **Verifica que el Service Provider esté registrado:**
   Revisa `bootstrap/providers.php`.

2. **Limpia la caché:**
   ```bash
   php artisan config:clear
   php artisan cache:clear
   ```

3. **Verifica que estés en modo consola:**
   Los comandos solo se registran cuando Laravel se ejecuta en consola.

## 📝 Programar Limpieza Automática

Puedes programar los comandos para que se ejecuten automáticamente usando el scheduler de Laravel.

En `app/Console/Kernel.php`:

```php
use Illuminate\Console\Scheduling\Schedule;

protected function schedule(Schedule $schedule): void
{
    // Limpiar soft-deleted cada semana
    $schedule->command('maintenance:clean-soft-deleted "App\Models\User" --days=30 --force')
             ->weekly()
             ->sundays()
             ->at('02:00');
    
    // Limpiar logs antiguos cada mes
    $schedule->command('maintenance:clean-old-records logs created_at --days=90 --force')
             ->monthly()
             ->at('03:00');
}
```

Recuerda configurar el cron en tu servidor:

```bash
* * * * * cd /path-to-your-project && php artisan schedule:run >> /dev/null 2>&1
```

## ⚠️ Advertencias Importantes

- **Backups:** Siempre haz backups antes de ejecutar operaciones de limpieza en producción
- **Pruebas:** Prueba los comandos en un entorno de desarrollo primero
- **Confirmaciones:** Por defecto, los comandos piden confirmación. Solo usa `--force` cuando estés seguro
- **No automático:** Este bloque NO ejecuta acciones automáticamente por defecto

## 📚 Documentación Adicional

Para más información, consulta:
- `README.md` - Documentación completa del bloque
- [Laravel Artisan Commands](https://laravel.com/docs/artisan) - Documentación oficial de Laravel
- [Laravel Task Scheduling](https://laravel.com/docs/scheduling) - Documentación oficial de Laravel

## 🎯 Próximos Pasos

Una vez instalado el bloque:

1. Verifica que los comandos estén disponibles
2. Prueba los comandos en desarrollo
3. Configura los comandos en tu scheduler si es necesario
4. ¡Disfruta de herramientas de mantenimiento limpias y seguras!

---

**¿Necesitas ayuda?** Consulta la documentación de Laravel sobre [Artisan Commands](https://laravel.com/docs/artisan) y [Task Scheduling](https://laravel.com/docs/scheduling).
