<?php

namespace App\Blocks\Maintenance\Services;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class CleanupService
{
    /**
     * Clean soft-deleted records older than specified days.
     */
    public function cleanSoftDeleted(string $modelClass, int $days = 30): int
    {
        if (!is_subclass_of($modelClass, Model::class)) {
            throw new \InvalidArgumentException("Class {$modelClass} must be an Eloquent Model.");
        }

        $model = new $modelClass();
        
        if (!method_exists($model, 'onlyTrashed')) {
            throw new \InvalidArgumentException("Model {$modelClass} must use SoftDeletes trait.");
        }

        $cutoffDate = now()->subDays($days);

        return $modelClass::onlyTrashed()
            ->where('deleted_at', '<', $cutoffDate)
            ->forceDelete();
    }

    /**
     * Clean orphaned records (records without valid foreign key relationships).
     */
    public function cleanOrphaned(string $modelClass, string $relationName, ?string $parentTable = null): int
    {
        if (!is_subclass_of($modelClass, Model::class)) {
            throw new \InvalidArgumentException("Class {$modelClass} must be an Eloquent Model.");
        }

        $model = new $modelClass();
        $relation = $model->$relationName();

        if (!$relation instanceof \Illuminate\Database\Eloquent\Relations\BelongsTo) {
            throw new \InvalidArgumentException("Relation {$relationName} must be a BelongsTo relationship.");
        }

        $foreignKey = $relation->getForeignKeyName();
        $parentKey = $relation->getOwnerKeyName();
        $parentTable = $parentTable ?? $relation->getRelated()->getTable();

        $deleted = DB::table($model->getTable())
            ->leftJoin($parentTable, "{$model->getTable()}.{$foreignKey}", '=', "{$parentTable}.{$parentKey}")
            ->whereNull("{$parentTable}.{$parentKey}")
            ->delete();

        return $deleted;
    }

    /**
     * Clean records from a table older than specified days.
     */
    public function cleanOldRecords(string $table, string $dateColumn, int $days = 30): int
    {
        $cutoffDate = now()->subDays($days);

        return DB::table($table)
            ->where($dateColumn, '<', $cutoffDate)
            ->delete();
    }

    /**
     * Clean records from a table by condition.
     */
    public function cleanByCondition(string $table, array $conditions): int
    {
        $query = DB::table($table);

        foreach ($conditions as $column => $value) {
            $query->where($column, $value);
        }

        return $query->delete();
    }
}
