<?php

namespace App\Blocks\Maintenance\Commands;

use App\Blocks\Maintenance\Services\CleanupService;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Model;

class CleanSoftDeletedCommand extends Command
{
    /**
     * The name and signature of the console command.
     */
    protected $signature = 'maintenance:clean-soft-deleted 
                            {model : The model class name (e.g., App\\Models\\User)}
                            {--days=30 : Number of days to keep soft-deleted records}
                            {--force : Force the operation without confirmation}';

    /**
     * The console command description.
     */
    protected $description = 'Clean soft-deleted records older than specified days';

    /**
     * Execute the console command.
     */
    public function handle(CleanupService $cleanupService): int
    {
        $modelClass = $this->argument('model');
        $days = (int) $this->option('days');
        $force = $this->option('force');

        // Validate model class
        if (!class_exists($modelClass)) {
            $this->error("Class {$modelClass} does not exist.");
            return Command::FAILURE;
        }

        if (!is_subclass_of($modelClass, Model::class)) {
            $this->error("Class {$modelClass} must be an Eloquent Model.");
            return Command::FAILURE;
        }

        // Confirm action
        if (!$force && !$this->confirm("This will permanently delete soft-deleted {$modelClass} records older than {$days} days. Continue?")) {
            $this->info('Operation cancelled.');
            return Command::SUCCESS;
        }

        try {
            $this->info("Cleaning soft-deleted records from {$modelClass}...");
            
            $deleted = $cleanupService->cleanSoftDeleted($modelClass, $days);
            
            $this->info("✓ Successfully deleted {$deleted} record(s).");
            
            return Command::SUCCESS;
        } catch (\Exception $e) {
            $this->error("Error: {$e->getMessage()}");
            return Command::FAILURE;
        }
    }
}
