<?php

namespace App\Blocks\Maintenance\Commands;

use App\Blocks\Maintenance\Services\CleanupService;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Model;

class CleanOrphanedCommand extends Command
{
    /**
     * The name and signature of the console command.
     */
    protected $signature = 'maintenance:clean-orphaned 
                            {model : The model class name (e.g., App\\Models\\Comment)}
                            {relation : The relation name (e.g., user)}
                            {--parent-table= : The parent table name (optional, will be auto-detected)}
                            {--force : Force the operation without confirmation}';

    /**
     * The console command description.
     */
    protected $description = 'Clean orphaned records (records without valid foreign key relationships)';

    /**
     * Execute the console command.
     */
    public function handle(CleanupService $cleanupService): int
    {
        $modelClass = $this->argument('model');
        $relationName = $this->argument('relation');
        $parentTable = $this->option('parent-table');
        $force = $this->option('force');

        // Validate model class
        if (!class_exists($modelClass)) {
            $this->error("Class {$modelClass} does not exist.");
            return Command::FAILURE;
        }

        if (!is_subclass_of($modelClass, Model::class)) {
            $this->error("Class {$modelClass} must be an Eloquent Model.");
            return Command::FAILURE;
        }

        // Confirm action
        if (!$force && !$this->confirm("This will permanently delete orphaned records from {$modelClass} (relation: {$relationName}). Continue?")) {
            $this->info('Operation cancelled.');
            return Command::SUCCESS;
        }

        try {
            $this->info("Cleaning orphaned records from {$modelClass}...");
            
            $deleted = $cleanupService->cleanOrphaned($modelClass, $relationName, $parentTable);
            
            $this->info("✓ Successfully deleted {$deleted} orphaned record(s).");
            
            return Command::SUCCESS;
        } catch (\Exception $e) {
            $this->error("Error: {$e->getMessage()}");
            return Command::FAILURE;
        }
    }
}
