<?php

namespace App\Blocks\Maintenance\Commands;

use App\Blocks\Maintenance\Services\CleanupService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class CleanOldRecordsCommand extends Command
{
    /**
     * The name and signature of the console command.
     */
    protected $signature = 'maintenance:clean-old-records 
                            {table : The table name}
                            {date-column : The date column name (e.g., created_at)}
                            {--days=30 : Number of days to keep records}
                            {--force : Force the operation without confirmation}';

    /**
     * The console command description.
     */
    protected $description = 'Clean records from a table older than specified days';

    /**
     * Execute the console command.
     */
    public function handle(CleanupService $cleanupService): int
    {
        $table = $this->argument('table');
        $dateColumn = $this->argument('date-column');
        $days = (int) $this->option('days');
        $force = $this->option('force');

        // Validate table exists
        if (!DB::getSchemaBuilder()->hasTable($table)) {
            $this->error("Table {$table} does not exist.");
            return Command::FAILURE;
        }

        // Confirm action
        if (!$force && !$this->confirm("This will permanently delete records from {$table} older than {$days} days (based on {$dateColumn}). Continue?")) {
            $this->info('Operation cancelled.');
            return Command::SUCCESS;
        }

        try {
            $this->info("Cleaning old records from {$table}...");
            
            $deleted = $cleanupService->cleanOldRecords($table, $dateColumn, $days);
            
            $this->info("✓ Successfully deleted {$deleted} record(s).");
            
            return Command::SUCCESS;
        } catch (\Exception $e) {
            $this->error("Error: {$e->getMessage()}");
            return Command::FAILURE;
        }
    }
}
