# Guía de Instalación - In-App Notifications Block (Light)

Esta guía te ayudará a integrar el bloque In-App Notifications en tu proyecto Laravel 12.

## 📋 Requisitos Previos

- Laravel 12.x instalado
- PHP 8.3 o superior
- Tailwind CSS configurado en tu proyecto
- Alpine.js instalado (para el dropdown)
- Sistema de autenticación configurado
- Tabla `users` existente en la base de datos

## 🚀 Pasos de Instalación

### Paso 1: Extraer el contenido

**Importante:** Descomprime el archivo ZIP en la **raíz de tu proyecto Laravel** (donde están las carpetas `app/`, `config/`, `routes/`, `bootstrap/`, `database/`, `resources/`, etc.).

Al descomprimir el ZIP, la estructura de archivos quedará así:

```
tu-proyecto-laravel/
├── app/
│   └── Blocks/
│       └── Notifications/              ← El bloque queda aquí
│           ├── Models/
│           ├── Services/
│           ├── Controllers/
│           ├── routes.php
│           └── NotificationsServiceProvider.php
├── database/
│   └── migrations/
│       └── 2024_01_01_000000_create_notifications_table.php  ← Migración
├── resources/
│   └── views/
│       └── components/
│           └── notifications.blade.php  ← Componente Blade
├── README.md
├── LICENSE.txt
└── docs/
    └── INSTALLATION.md
```

**Nota:** Asegúrate de descomprimir el ZIP en la raíz del proyecto, no dentro de una subcarpeta. Los archivos deben quedar directamente en las carpetas `app/`, `database/`, `resources/`, etc.

### Paso 2: Ejecutar la migración

Ejecuta la migración para crear la tabla `notifications`:

```bash
php artisan migrate
```

Esto creará la tabla `notifications` con los siguientes campos:
- `id` - Identificador único
- `user_id` - ID del usuario (foreign key)
- `title` - Título de la notificación
- `body` - Contenido de la notificación
- `read_at` - Fecha de lectura (nullable)
- `created_at` y `updated_at` - Timestamps

### Paso 3: Registrar el Service Provider

Abre el archivo `bootstrap/providers.php` y añade el Service Provider del bloque:

```php
<?php

return [
    App\Providers\AppServiceProvider::class,
    App\Blocks\Notifications\NotificationsServiceProvider::class, // ← Añadir esta línea
];
```

### Paso 4: JavaScript (opcional)

El componente incluye JavaScript vanilla simple integrado para el dropdown. No requiere dependencias externas como Alpine.js. El script está incluido directamente en el componente Blade.

### Paso 5: Incluir el componente en tu layout

Abre tu layout principal (generalmente `resources/views/layouts/app.blade.php`) y agrega el componente en el navbar, dentro de la sección de usuario autenticado:

```blade
<nav>
    <!-- ... otros elementos ... -->
    
    @auth
        <x-notifications />
        <!-- Otros elementos del navbar para usuarios autenticados -->
    @endauth
</nav>
```

### Paso 6: Limpiar caché (opcional)

```bash
php artisan config:clear
php artisan view:clear
php artisan route:clear
```

## ✅ Verificación

Para verificar que todo funciona correctamente:

1. **Verifica la migración:**
   ```bash
   php artisan migrate:status
   ```
   Debe mostrar la migración `create_notifications_table` como ejecutada.

2. **Crea una notificación de prueba:**
   ```php
   use App\Blocks\Notifications\Services\NotificationService;
   
   NotificationService::send(
       user: auth()->user(),
       title: 'Notificación de prueba',
       body: 'Esta es una notificación de prueba'
   );
   ```

3. **Verifica el componente:**
   - Inicia sesión en tu aplicación
   - Deberías ver un icono de campana en el navbar
   - Haz clic en el icono para ver el dropdown de notificaciones

## 🔧 Solución de Problemas

### El componente no se muestra

1. **Verifica que el usuario esté autenticado:**
   El componente solo se muestra para usuarios autenticados (`@auth`).

2. **Verifica la ubicación del archivo:**
   ```bash
   ls -la resources/views/components/notifications.blade.php
   ```

3. **Limpia el cache de vistas:**
   ```bash
   php artisan view:clear
   ```

### El dropdown no funciona

1. **Verifica que el JavaScript esté cargado:**
   El componente incluye su propio script JavaScript vanilla. Asegúrate de que no haya errores en la consola del navegador.

2. **Verifica que no haya conflictos de JavaScript:**
   Si tienes otros scripts que puedan interferir, revisa la consola del navegador para errores.

### Error: "Class not found"

1. **Verifica que el Service Provider esté registrado:**
   Revisa `bootstrap/providers.php`.

2. **Limpia el cache:**
   ```bash
   php artisan config:clear
   php artisan cache:clear
   ```

### La migración falla

1. **Verifica que la tabla `users` exista:**
   ```bash
   php artisan tinker
   >>> Schema::hasTable('users')
   ```

2. **Verifica que no exista la tabla `notifications`:**
   Si ya existe, puedes eliminarla y volver a ejecutar la migración, o modificar la migración para que no falle.

## 📝 Uso Básico

### Crear notificaciones desde un controlador

```php
use App\Blocks\Notifications\Services\NotificationService;

class OrderController extends Controller
{
    public function store(Request $request)
    {
        $order = Order::create($request->validated());
        
        // Notificar al usuario
        NotificationService::send(
            user: auth()->user(),
            title: 'Orden creada',
            body: "Tu orden #{$order->id} ha sido creada exitosamente."
        );
        
        return redirect()->route('orders.index')
            ->with('success', 'Orden creada exitosamente.');
    }
}
```

### Crear notificaciones desde un evento

```php
use App\Blocks\Notifications\Services\NotificationService;

class OrderCreated
{
    public function handle(OrderCreated $event)
    {
        NotificationService::send(
            user: $event->order->user,
            title: 'Nueva orden',
            body: "Has recibido una nueva orden #{$event->order->id}"
        );
    }
}
```

### Obtener notificaciones manualmente

```php
use App\Blocks\Notifications\Services\NotificationService;

// Obtener notificaciones
$notifications = NotificationService::forUser(auth()->user(), 10);

// Obtener conteo de no leídas
$unreadCount = NotificationService::unreadCount(auth()->user());
```

## 🎨 Personalización

### Cambiar el límite de notificaciones

Edita `resources/views/components/notifications.blade.php`:

```php
$notifications = NotificationService::forUser(auth()->user(), 20); // Cambiar 10 a 20
```

### Cambiar los estilos

El componente usa Tailwind CSS. Puedes modificar las clases directamente en `resources/views/components/notifications.blade.php`.

### Cambiar la posición del dropdown

Modifica las clases `absolute right-0` en el componente para cambiar la posición.

## 📚 Documentación Adicional

Para más información, consulta:
- `README.md` - Documentación completa del bloque
- `LICENSE.txt` - Términos de licencia

## 🎯 Próximos Pasos

Una vez instalado el bloque:

1. Integra el componente en tu layout principal
2. Comienza a crear notificaciones desde tus controladores
3. Personaliza los estilos según tus necesidades
4. ¡Disfruta de notificaciones internas elegantes en tu aplicación!

---

**¿Necesitas ayuda?** Consulta la documentación de Laravel sobre [autenticación](https://laravel.com/docs/authentication) y [migraciones](https://laravel.com/docs/migrations).
