# In-App Notifications Block (Light) - Laravel 12

Un bloque de notificaciones internas (in-app) simple y elegante para Laravel 12, diseñado para mostrar notificaciones a usuarios autenticados mediante base de datos.

## 📋 Descripción

Este **In-App Notifications Block (Light)** proporciona un sistema básico de notificaciones internas con:

- Almacenamiento en base de datos
- Visualización de notificaciones para usuarios autenticados
- Marcado explícito de notificaciones como leídas
- Diseño elegante con Tailwind CSS
- Componente Blade reutilizable
- Arquitectura limpia y fácil de extender

**Importante:** Este es un bloque LIGHT y gratuito. Está diseñado para ser integrado fácilmente en proyectos Laravel 12. No incluye emails, SMS, WebSockets, colas ni funcionalidades avanzadas.

## 🎯 Características

- ✅ Guardar notificaciones en base de datos
- ✅ Mostrar notificaciones del usuario autenticado
- ✅ Marcar notificaciones como leídas (individual y masivo)
- ✅ Diferenciación visual entre leídas y no leídas
- ✅ Contador de notificaciones no leídas
- ✅ Dropdown elegante con Tailwind CSS
- ✅ Compatible con dark mode
- ✅ Sin dependencias externas
- ✅ Arquitectura limpia (Model, Service, Controller)

## 📦 Requisitos

- Laravel 12.x
- PHP 8.3 o superior
- Tailwind CSS configurado en el proyecto
- Tabla `users` existente en la base de datos
- Sistema de autenticación configurado

## 🚀 Instalación

### 1. Copiar archivos

Copia la carpeta `app/Blocks/Notifications/` a tu proyecto Laravel en la misma ubicación.

### 2. Copiar migración

Copia el archivo de migración a `database/migrations/` de tu proyecto.

### 3. Ejecutar migración

```bash
php artisan migrate
```

### 4. Copiar componente Blade

Copia `resources/views/components/notifications.blade.php` a tu proyecto en la misma ubicación.

### 5. Registrar Service Provider

Añade el Service Provider en `bootstrap/providers.php`:

```php
<?php

return [
    App\Providers\AppServiceProvider::class,
    App\Blocks\Notifications\NotificationsServiceProvider::class, // ← Añadir esta línea
];
```

### 6. Incluir componente en tu layout

Abre tu layout principal (ej: `resources/views/layouts/app.blade.php`) y agrega el componente en el navbar:

```blade
<nav>
    <!-- ... otros elementos del navbar ... -->
    
    @auth
        <x-notifications />
    @endauth
</nav>
```

### 7. JavaScript (opcional)

El componente incluye JavaScript vanilla simple integrado para el dropdown. No requiere dependencias externas como Alpine.js.

## 💡 Uso

### Crear notificaciones

Desde tus controladores o servicios:

```php
use App\Blocks\Notifications\Services\NotificationService;
use App\Models\User;

// Crear una notificación
$user = User::find(1);
NotificationService::send(
    user: $user,
    title: 'Nueva orden recibida',
    body: 'Has recibido una nueva orden #12345'
);
```

### Ejemplo completo

```php
use App\Blocks\Notifications\Services\NotificationService;

class OrderController extends Controller
{
    public function store(Request $request)
    {
        $order = Order::create($request->validated());
        
        // Notificar al usuario
        NotificationService::send(
            user: auth()->user(),
            title: 'Orden creada',
            body: "Tu orden #{$order->id} ha sido creada exitosamente."
        );
        
        return redirect()->route('orders.index')
            ->with('success', 'Orden creada exitosamente.');
    }
}
```

### Obtener notificaciones

El componente `<x-notifications />` se encarga automáticamente de:
- Obtener las notificaciones del usuario autenticado
- Mostrar el contador de no leídas
- Permitir marcar como leídas

## 🎨 Personalización

### Cambiar el límite de notificaciones mostradas

Edita `resources/views/components/notifications.blade.php`:

```php
$notifications = NotificationService::forUser(auth()->user(), 20); // Cambiar 10 a 20
```

### Cambiar los estilos

El componente usa Tailwind CSS. Puedes modificar las clases directamente en `resources/views/components/notifications.blade.php`.

## 📁 Estructura del Bloque

```
in-app-notifications-laravel12/
├── app/
│   └── Blocks/
│       └── Notifications/
│           ├── Models/
│           │   └── Notification.php
│           ├── Services/
│           │   └── NotificationService.php
│           ├── Controllers/
│           │   └── NotificationController.php
│           ├── routes.php
│           └── NotificationsServiceProvider.php
├── database/
│   └── migrations/
│       └── 2024_01_01_000000_create_notifications_table.php
├── resources/
│   └── views/
│       └── components/
│           └── notifications.blade.php
├── docs/
│   └── INSTALLATION.md
├── README.md
└── LICENSE.txt
```

## 🔧 Notas Técnicas

- Las notificaciones se almacenan en la tabla `notifications`
- Solo usuarios autenticados pueden ver sus notificaciones
- El marcado como leída es explícito (no automático)
- El componente requiere Alpine.js para el dropdown
- Compatible con la autenticación estándar de Laravel

## ⚠️ Limitaciones (Bloque Light)

Este bloque NO incluye:
- Envío de emails
- SMS
- Broadcast
- WebSockets
- Colas
- Redis
- Preferencias de notificación
- Templates avanzados
- Panel administrativo
- Roles o permisos

## 📝 Métodos del Service

### NotificationService::send()
Crea una nueva notificación para un usuario.

### NotificationService::unreadCount()
Obtiene el conteo de notificaciones no leídas de un usuario.

### NotificationService::forUser()
Obtiene las notificaciones de un usuario (con límite opcional).

## 🎯 Compatibilidad

- Laravel 12.x
- PHP 8.3+
- Tailwind CSS 3.x o superior
- JavaScript vanilla (incluido en el componente, sin dependencias externas)

## 📄 Licencia

Este bloque es gratuito y está disponible bajo la licencia MIT. Ver `LICENSE.txt` para más detalles.

---

**In-App Notifications Block (Light) v1.0.0** - Laravel 12
