# Authentication Block - Laravel 12

Un bloque de autenticación reutilizable y elegante para Laravel 12, diseñado para ser integrado fácilmente en proyectos nuevos o existentes.

## 📋 Descripción

Este **Authentication Block** proporciona un sistema de autenticación completo (login y registro) con:
- Vista de login estilizada con Tailwind CSS
- Vista de registro con validación
- Dashboard simple post-autenticación
- Arquitectura limpia y modular
- Componentes Blade reutilizables
- Validación mediante Form Requests
- Lógica de negocio en Services

**Importante:** Este es un bloque, no una aplicación completa. Está diseñado para ser integrado en proyectos Laravel 12.

## 📸 Capturas de Pantalla

El bloque incluye capturas de pantalla en la carpeta `screenshots/` que muestran:
- Vista de login
- Vista de registro  
- Dashboard post-autenticación
- Diseño y componentes

Estas imágenes pueden ser usadas para documentación, demostraciones o marketing del producto.

## 🎯 Características

- ✅ Sistema de login funcional
- ✅ Sistema de registro con validación
- ✅ Validación de formularios con rate limiting
- ✅ Componentes Blade reutilizables (Input, Button, Layout)
- ✅ Diseño moderno y profesional con Tailwind CSS
- ✅ Dashboard básico post-autenticación
- ✅ Arquitectura limpia (Controllers, Services, Requests)
- ✅ Rutas aisladas con prefijo `/auth`
- ✅ Service Provider propio del bloque

## 📦 Requisitos

- Laravel 12.x
- PHP 8.3 o superior
- Tailwind CSS configurado en el proyecto
- Tabla `users` existente en la base de datos

## 🚀 Instalación

### 1. Copiar archivos

Copia toda la carpeta `app/Blocks/Auth/` a tu proyecto Laravel.

### 2. Registrar Service Provider

Añade el Service Provider en `bootstrap/providers.php`:

```php
<?php

return [
    App\Providers\AppServiceProvider::class,
    App\Blocks\Auth\AuthServiceProvider::class, // ← Añadir esta línea
];
```

### 3. Verificar configuración de autenticación

Asegúrate de que `config/auth.php` tenga la configuración estándar de Laravel:

```php
'guards' => [
    'web' => [
        'driver' => 'session',
        'provider' => 'users',
    ],
],

'providers' => [
    'users' => [
        'driver' => 'eloquent',
        'model' => App\Models\User::class,
    ],
],
```

### 4. Verificar Tailwind CSS

Asegúrate de que Tailwind CSS esté configurado en tu proyecto. El bloque usa clases estándar de Tailwind sin plugins externos.

### 5. Limpiar caché (opcional)

```bash
php artisan config:clear
php artisan view:clear
php artisan route:clear
```

## 🎨 Rutas Disponibles

El bloque registra las siguientes rutas con el prefijo `/auth`:

- `GET /auth/login` - Muestra el formulario de login
- `POST /auth/login` - Procesa el login
- `GET /auth/register` - Muestra el formulario de registro
- `POST /auth/register` - Procesa el registro
- `POST /auth/logout` - Cierra la sesión
- `GET /auth/dashboard` - Dashboard (requiere autenticación)

## 📁 Estructura del Bloque

```
app/Blocks/Auth/
├── AuthServiceProvider.php    # Service Provider del bloque
├── routes.php                  # Rutas aisladas del bloque
├── Controllers/
│   └── AuthController.php     # Controlador de autenticación
├── Requests/
│   ├── LoginRequest.php       # Form Request para validación de login
│   └── RegisterRequest.php    # Form Request para validación de registro
├── Services/
│   └── AuthService.php        # Lógica de autenticación
└── Views/
    ├── Components/            # Componentes Blade de clase
    │   ├── Input.php
    │   ├── Button.php
    │   └── Layout.php
    ├── components/             # Templates Blade de componentes
    │   ├── input.blade.php
    │   ├── button.blade.php
    │   └── layout.blade.php
    ├── login.blade.php        # Vista de login
    ├── register.blade.php     # Vista de registro
    └── dashboard.blade.php    # Vista de dashboard
```

## 🎨 Componentes Blade

El bloque incluye componentes Blade reutilizables:

### Layout Component

```blade
<x-auth-block::layout title="Mi Título">
    <!-- Contenido -->
</x-auth-block::layout>
```

### Input Component

```blade
<x-auth-block::input
    name="email"
    type="email"
    placeholder="tu@email.com"
    required
/>
```

### Button Component

```blade
<x-auth-block::button 
    type="submit" 
    variant="primary" 
    full-width
>
    Enviar
</x-auth-block::button>
```

Variantes disponibles: `primary`, `secondary`, `danger`

## 🔧 Personalización

### Cambiar prefijo de rutas

Edita `app/Blocks/Auth/AuthServiceProvider.php`:

```php
Route::middleware('web')
    ->prefix('tu-prefijo') // Cambiar aquí
    ->group(base_path('app/Blocks/Auth/routes.php'));
```

### Personalizar vistas

Las vistas están en `app/Blocks/Auth/Views/`. Puedes modificarlas directamente o publicarlas:

```php
// En AuthServiceProvider::boot()
$this->publishes([
    base_path('app/Blocks/Auth/Views') => resource_path('views/vendor/auth-block'),
], 'auth-block-views');
```

### Personalizar componentes

Los componentes están en `app/Blocks/Auth/Views/Components/` y sus templates en `app/Blocks/Auth/Views/components/`.

## 🎯 Uso Básico

### Registro
1. Accede a `/auth/register` para ver el formulario de registro
2. Completa el formulario con nombre, email y contraseña
3. Tras el registro exitoso, serás autenticado automáticamente y redirigido a `/auth/dashboard`

### Login
1. Accede a `/auth/login` para ver el formulario de login
2. Ingresa las credenciales de un usuario existente
3. Tras el login exitoso, serás redirigido a `/auth/dashboard`
4. Usa el botón "Cerrar Sesión" para salir

## 📝 Notas Importantes

- **Incluye registro de usuarios**: Login y registro completos
- **No incluye recuperación de contraseña**: Solo login y registro
- **No incluye roles ni permisos**: Solo autenticación básica
- **Asume tabla `users` existente**: No crea migraciones
- **Dashboard básico**: Solo muestra mensaje de bienvenida
- **Validación de contraseña**: Usa las reglas de validación estándar de Laravel

## 🔒 Seguridad

- Rate limiting en login (5 intentos por minuto por IP)
- Validación de CSRF en todos los formularios
- Regeneración de sesión tras login exitoso
- Validación de credenciales con mensajes de error claros

## 🎨 Diseño

El bloque usa un diseño limpio y profesional:
- Fondo claro (gray-50)
- Inputs accesibles con estados de error
- Botones sobrios y modernos
- Tipografía legible
- Responsive design

## 📄 Licencia y Términos de Uso

### Licencia de Uso

Este bloque de autenticación está diseñado para ser vendido como script individual. Al adquirir este producto, obtienes una licencia de uso personal o comercial para un solo proyecto.

### Restricciones Importantes

**❌ PROHIBIDO:**
- Revender, redistribuir o compartir este código
- Incluir este código en otros productos o scripts para venta
- Modificar y vender como producto propio
- Compartir el código con terceros sin autorización
- Usar en múltiples proyectos sin licencias adicionales

**✅ PERMITIDO:**
- Usar en un proyecto personal o comercial
- Modificar el código para tus necesidades
- Integrar en aplicaciones cliente
- Usar como base para proyectos propios (sin revender el código)

### Uso Comercial

Puedes usar este bloque en proyectos comerciales y aplicaciones cliente sin restricciones, siempre que no revendas o redistribuyas el código fuente.

**Al usar este código, aceptas estos términos de licencia.**

## 🤝 Soporte

**Importante sobre Soporte:**

- **Scripts de Pago:** Incluyen soporte según los términos de tu compra. Consulta los detalles de soporte en tu plataforma de compra.
- **Scripts Gratuitos:** NO incluyen soporte técnico. El código se proporciona "tal cual" sin garantías ni asistencia.
- **Documentación:** Para personalizaciones o integraciones específicas, consulta la documentación de Laravel 12.
- **Licencias:** Para consultas sobre licencias, consulta el archivo `LICENSE.txt` incluido.

---

**Desarrollado para Laravel 12** | Authentication Block v1.0
