# Guía de Instalación - Activity Timestamp Helpers Block (Light)

Esta guía te ayudará a integrar el bloque Activity Timestamp Helpers en tu proyecto Laravel 12.

## 📋 Requisitos Previos

- Laravel 12.x instalado
- PHP 8.3 o superior
- Carbon (incluido con Laravel)

## 🚀 Pasos de Instalación

### Paso 1: Extraer el contenido

**Importante:** Descomprime el archivo ZIP en la **raíz de tu proyecto Laravel** (donde están las carpetas `app/`, `config/`, `routes/`, `bootstrap/`, etc.).

Al descomprimir el ZIP, la estructura de archivos quedará así:

```
tu-proyecto-laravel/
├── app/
│   └── Blocks/
│       └── Timestamps/              ← El bloque queda aquí
│           ├── Services/
│           ├── Components/
│           ├── Views/
│           └── TimestampServiceProvider.php
├── README.md
├── LICENSE.txt
└── docs/
    └── INSTALLATION.md
```

**Nota:** Asegúrate de descomprimir el ZIP en la raíz del proyecto, no dentro de una subcarpeta. Los archivos deben quedar directamente en las carpetas `app/`, etc.

### Paso 2: Registrar el Service Provider

Abre el archivo `bootstrap/providers.php` y añade el Service Provider del bloque:

```php
<?php

return [
    App\Providers\AppServiceProvider::class,
    App\Blocks\Timestamps\TimestampServiceProvider::class, // ← Añadir esta línea
];
```

### Paso 3: Limpiar caché (opcional)

```bash
php artisan config:clear
php artisan view:clear
```

## ✅ Verificación

Para verificar que todo funciona correctamente:

1. **Usa el componente en una vista:**
   ```blade
   @php
       $date = now()->subHours(2);
   @endphp
   
   <x-timestamp-block::timestamp :datetime="$date" />
   ```
   
   Deberías ver "2 hours ago" o similar.

2. **Usa el servicio en PHP:**
   ```php
   use App\Blocks\Timestamps\Services\TimestampService;
   
   $text = TimestampService::humanDiff(now()->subHours(2));
   echo $text; // "2 hours ago"
   ```

## 🔧 Ejemplos de Uso

### Usar el componente Blade

```blade
{{-- Formato humano (default) --}}
<x-timestamp-block::timestamp :datetime="$post->created_at" />

{{-- Formato corto --}}
<x-timestamp-block::timestamp :datetime="$post->created_at" format="short" />

{{-- Formato largo --}}
<x-timestamp-block::timestamp :datetime="$post->created_at" format="long" />

{{-- Con etiqueta Today/Yesterday --}}
<x-timestamp-block::timestamp :datetime="$post->created_at" format="with-label" />
```

### Usar el servicio directamente

```php
use App\Blocks\Timestamps\Services\TimestampService;

// En un controlador o cualquier clase
$humanDiff = TimestampService::humanDiff($post->created_at);
$shortDate = TimestampService::short($post->created_at);
$longDate = TimestampService::long($post->created_at);
```

### Ejemplo completo: Lista de posts

```blade
@foreach($posts as $post)
    <article class="border-b py-4">
        <h2 class="text-xl font-bold">{{ $post->title }}</h2>
        <p class="text-sm text-gray-600">
            Publicado <x-timestamp-block::timestamp :datetime="$post->created_at" />
        </p>
        <p>{{ $post->excerpt }}</p>
    </article>
@endforeach
```

### Ejemplo completo: Card con múltiples timestamps

```blade
<div class="card">
    <h3>{{ $article->title }}</h3>
    
    <div class="text-sm text-gray-600 space-y-1">
        <div>
            Publicado: <x-timestamp-block::timestamp :datetime="$article->published_at" format="long" />
        </div>
        <div>
            Actualizado: <x-timestamp-block::timestamp :datetime="$article->updated_at" format="human" />
        </div>
    </div>
</div>
```

## 🔧 Solución de Problemas

### Error: "Class not found"

1. **Verifica que los archivos estén en la ubicación correcta:**
   ```bash
   ls -la app/Blocks/Timestamps/Services/
   ```

2. **Ejecuta composer dump-autoload:**
   ```bash
   composer dump-autoload
   ```

3. **Verifica que el Service Provider esté registrado:**
   Revisa `bootstrap/providers.php`.

### El componente no se muestra

1. **Verifica que el Service Provider esté registrado:**
   El Service Provider es quien registra el componente Blade.

2. **Limpia la caché de vistas:**
   ```bash
   php artisan view:clear
   php artisan config:clear
   ```

3. **Verifica que el datetime sea válido:**
   ```blade
   @php
       // Asegúrate de que $datetime sea válido
       dump($post->created_at);
   @endphp
   ```

### Los formatos no cambian

1. **Verifica que estés usando el formato correcto:**
   Los formatos disponibles son: `human`, `short`, `long`, `with-label`, `time`, `date`

2. **Limpia la caché:**
   ```bash
   php artisan view:clear
   ```

### Las fechas están en inglés cuando quieres español

El bloque usa Carbon que respeta la configuración de locale de Laravel. Verifica `config/app.php`:

```php
'locale' => 'es', // o 'en'
```

Luego ejecuta:
```bash
php artisan config:clear
```

## 📝 Personalización

### Personalizar estilos

Puedes agregar clases CSS al componente:

```blade
<x-timestamp-block::timestamp 
    :datetime="$post->created_at" 
    class="font-semibold text-blue-600 hover:underline" 
/>
```

### Crear formatos personalizados

Puedes extender `TimestampService` o crear métodos personalizados:

```php
use App\Blocks\Timestamps\Services\TimestampService;

class CustomTimestampService extends TimestampService
{
    public static function customFormat($datetime): string
    {
        $date = self::toCarbon($datetime);
        return $date->format('d/m/Y H:i');
    }
}
```

### Personalizar el componente

Si necesitas personalizar el componente, puedes:

1. **Publicar la vista:**
   Copia `app/Blocks/Timestamps/Views/components/timestamp.blade.php` a `resources/views/vendor/timestamp-block/components/timestamp.blade.php`

2. **Modificar directamente:**
   Edita `app/Blocks/Timestamps/Views/components/timestamp.blade.php` directamente

## 📚 Documentación Adicional

Para más información, consulta:
- `README.md` - Documentación completa del bloque
- [Carbon Documentation](https://carbon.nesbot.com/docs/) - Documentación oficial de Carbon
- [Laravel Localization](https://laravel.com/docs/localization) - Documentación oficial de Laravel

## 🎯 Próximos Pasos

Una vez instalado el bloque:

1. Usa el componente `<x-timestamp-block::timestamp />` en tus vistas
2. Usa `TimestampService` cuando necesites formatear fechas en PHP
3. Personaliza los estilos según tus necesidades
4. ¡Disfruta de timestamps consistentes y elegantes!

---

**¿Necesitas ayuda?** Consulta la documentación de [Carbon](https://carbon.nesbot.com/docs/) y [Laravel Localization](https://laravel.com/docs/localization).
