<?php

namespace App\Blocks\Timestamps\Services;

use Carbon\Carbon;

class TimestampService
{
    /**
     * Get human-readable time difference (e.g., "2 hours ago").
     */
    public static function humanDiff($datetime, ?Carbon $from = null): string
    {
        $date = self::toCarbon($datetime);
        $from = $from ?? now();

        return $date->diffForHumans($from);
    }

    /**
     * Get formatted date with label (Today, Yesterday, or date).
     */
    public static function withLabel($datetime, string $format = 'Y-m-d H:i'): string
    {
        $date = self::toCarbon($datetime);
        $today = now()->startOfDay();
        $yesterday = now()->subDay()->startOfDay();
        $dateStart = $date->copy()->startOfDay();

        if ($dateStart->equalTo($today)) {
            return 'Today, ' . $date->format($format);
        } elseif ($dateStart->equalTo($yesterday)) {
            return 'Yesterday, ' . $date->format($format);
        } else {
            return $date->format($format);
        }
    }

    /**
     * Get short format (e.g., "Jan 15, 2024").
     */
    public static function short($datetime): string
    {
        $date = self::toCarbon($datetime);
        return $date->format('M j, Y');
    }

    /**
     * Get long format (e.g., "January 15, 2024 at 3:45 PM").
     */
    public static function long($datetime): string
    {
        $date = self::toCarbon($datetime);
        return $date->format('F j, Y \a\t g:i A');
    }

    /**
     * Get time only (e.g., "3:45 PM").
     */
    public static function time($datetime): string
    {
        $date = self::toCarbon($datetime);
        return $date->format('g:i A');
    }

    /**
     * Get date only (e.g., "January 15, 2024").
     */
    public static function date($datetime): string
    {
        $date = self::toCarbon($datetime);
        return $date->format('F j, Y');
    }

    /**
     * Get ISO format for datetime attributes.
     */
    public static function iso($datetime): string
    {
        $date = self::toCarbon($datetime);
        return $date->toIso8601String();
    }

    /**
     * Convert various input types to Carbon instance.
     */
    protected static function toCarbon($datetime): Carbon
    {
        if ($datetime instanceof Carbon) {
            return $datetime;
        }

        if ($datetime instanceof \DateTime) {
            return Carbon::instance($datetime);
        }

        if (is_string($datetime)) {
            return Carbon::parse($datetime);
        }

        if (is_int($datetime)) {
            return Carbon::createFromTimestamp($datetime);
        }

        throw new \InvalidArgumentException('Invalid datetime value provided.');
    }
}
