# Activity Timestamp Helpers Block (Light) - Laravel 12

Un bloque de helpers y componentes reutilizables para mostrar fechas y tiempos de forma consistente, legible y elegante en proyectos Laravel 12.

## 📋 Descripción

Este **Activity Timestamp Helpers Block (Light)** proporciona helpers y componentes para mostrar fechas y tiempos de forma consistente:

- Formato amigable (ej. "2 hours ago")
- Formatos cortos y largos
- Etiquetas como "Today" / "Yesterday"
- Componente Blade reutilizable
- Estilos sobrios con Tailwind CSS

**Importante:** Este es un bloque LIGHT. NO incluye gestión avanzada de timezones, preferencias por usuario ni lógica de negocio compleja. Está diseñado para proporcionar consistencia visual y técnica en la presentación de fechas y tiempos.

## 🎯 Características

- ✅ Servicio `TimestampService` con múltiples métodos de formateo
- ✅ Componente Blade `<x-timestamp-block::timestamp />` reutilizable
- ✅ Formatos: human, short, long, with-label, time, date
- ✅ Tooltips con información adicional
- ✅ Compatible con Carbon y timestamps de Laravel
- ✅ Estilos sobrios con clases Tailwind CSS
- ✅ Sin dependencias externas (usa Carbon incluido en Laravel)

## 📦 Requisitos

- Laravel 12.x
- PHP 8.3 o superior
- Carbon (incluido con Laravel)

## 🚀 Instalación

### 1. Copiar archivos

Copia la carpeta `app/Blocks/Timestamps/` a tu proyecto Laravel en la misma ubicación.

### 2. Registrar Service Provider

Añade el Service Provider en `bootstrap/providers.php`:

```php
<?php

return [
    App\Providers\AppServiceProvider::class,
    App\Blocks\Timestamps\TimestampServiceProvider::class, // ← Añadir esta línea
];
```

### 3. Copiar componente Blade (opcional)

Si necesitas personalizar el componente, copia `resources/views/components/timestamp.blade.php` a tu proyecto.

### 4. Limpiar caché (opcional)

```bash
php artisan config:clear
php artisan view:clear
```

## 💡 Uso

### Usar el componente Blade

```blade
{{-- Formato humano (default) --}}
<x-timestamp-block::timestamp :datetime="$post->created_at" />

{{-- Formato corto --}}
<x-timestamp-block::timestamp :datetime="$post->created_at" format="short" />

{{-- Formato largo --}}
<x-timestamp-block::timestamp :datetime="$post->created_at" format="long" />

{{-- Con etiqueta Today/Yesterday --}}
<x-timestamp-block::timestamp :datetime="$post->created_at" format="with-label" />

{{-- Solo tiempo --}}
<x-timestamp-block::timestamp :datetime="$post->created_at" format="time" />

{{-- Solo fecha --}}
<x-timestamp-block::timestamp :datetime="$post->created_at" format="date" />
```

### Usar el servicio directamente

```php
use App\Blocks\Timestamps\Services\TimestampService;

// Formato humano
$text = TimestampService::humanDiff($post->created_at);
// "2 hours ago"

// Formato corto
$text = TimestampService::short($post->created_at);
// "Jan 15, 2024"

// Formato largo
$text = TimestampService::long($post->created_at);
// "January 15, 2024 at 3:45 PM"

// Con etiqueta
$text = TimestampService::withLabel($post->created_at);
// "Today, 2024-01-15 15:45" o "Yesterday, 2024-01-14 15:45"

// Solo tiempo
$text = TimestampService::time($post->created_at);
// "3:45 PM"

// Solo fecha
$text = TimestampService::date($post->created_at);
// "January 15, 2024"
```

## 📁 Estructura del Bloque

```
activity-timestamps-laravel12/
├── app/
│   └── Blocks/
│       └── Timestamps/
│           ├── Services/
│           │   └── TimestampService.php
│           ├── Components/
│           │   └── Timestamp.php
│           ├── Views/
│           │   └── components/
│           │       └── timestamp.blade.php
│           └── TimestampServiceProvider.php
├── resources/
│   └── views/
│       └── components/
│           └── timestamp.blade.php
├── docs/
│   └── INSTALLATION.md
├── README.md
└── LICENSE.txt
```

## 🔧 Formatos Disponibles

### human (default)

Muestra el tiempo relativo: "2 hours ago", "3 days ago", etc.

```blade
<x-timestamp-block::timestamp :datetime="$post->created_at" format="human" />
```

### short

Formato corto: "Jan 15, 2024"

```blade
<x-timestamp-block::timestamp :datetime="$post->created_at" format="short" />
```

### long

Formato largo: "January 15, 2024 at 3:45 PM"

```blade
<x-timestamp-block::timestamp :datetime="$post->created_at" format="long" />
```

### with-label

Con etiqueta Today/Yesterday: "Today, 2024-01-15 15:45"

```blade
<x-timestamp-block::timestamp :datetime="$post->created_at" format="with-label" />
```

### time

Solo tiempo: "3:45 PM"

```blade
<x-timestamp-block::timestamp :datetime="$post->created_at" format="time" />
```

### date

Solo fecha: "January 15, 2024"

```blade
<x-timestamp-block::timestamp :datetime="$post->created_at" format="date" />
```

## 📝 Ejemplos de Uso

### Ejemplo 1: Lista de posts

```blade
@foreach($posts as $post)
    <div class="border-b py-4">
        <h2>{{ $post->title }}</h2>
        <p class="text-sm">
            Publicado <x-timestamp-block::timestamp :datetime="$post->created_at" />
        </p>
    </div>
@endforeach
```

### Ejemplo 2: Card con fecha completa

```blade
<div class="card">
    <h3>{{ $article->title }}</h3>
    <p>
        <x-timestamp-block::timestamp :datetime="$article->published_at" format="long" />
    </p>
</div>
```

### Ejemplo 3: Usando el servicio en backend

```php
use App\Blocks\Timestamps\Services\TimestampService;

public function show($id)
{
    $post = Post::findOrFail($id);
    
    return view('posts.show', [
        'post' => $post,
        'publishedAgo' => TimestampService::humanDiff($post->created_at),
        'publishedDate' => TimestampService::date($post->created_at),
    ]);
}
```

### Ejemplo 4: Sin tooltip

```blade
<x-timestamp-block::timestamp :datetime="$post->created_at" :show-tooltip="false" />
```

### Ejemplo 5: Con clases personalizadas

```blade
<x-timestamp-block::timestamp 
    :datetime="$post->created_at" 
    class="font-semibold text-blue-600" 
/>
```

## 🔧 Métodos del TimestampService

- `humanDiff($datetime, $from = null)` - Tiempo relativo (ej: "2 hours ago")
- `withLabel($datetime, $format = 'Y-m-d H:i')` - Con etiqueta Today/Yesterday
- `short($datetime)` - Formato corto (ej: "Jan 15, 2024")
- `long($datetime)` - Formato largo (ej: "January 15, 2024 at 3:45 PM")
- `time($datetime)` - Solo tiempo (ej: "3:45 PM")
- `date($datetime)` - Solo fecha (ej: "January 15, 2024")
- `iso($datetime)` - Formato ISO para atributos HTML

## ⚠️ Notas Importantes

- El bloque usa Carbon (incluido con Laravel)
- Los formatos están optimizados para inglés/español según la configuración de Laravel
- El componente incluye atributos HTML `datetime` para accesibilidad
- Los tooltips muestran información complementaria al pasar el mouse
- Puedes personalizar los estilos agregando clases CSS

## 🎯 Compatibilidad

- Laravel 12.x
- PHP 8.3+
- Carbon 2.x (incluido con Laravel)

## 📄 Licencia

Este bloque es gratuito y está disponible bajo la licencia MIT. Ver `LICENSE.txt` para más detalles.

---

**Activity Timestamp Helpers Block (Light) v1.0.0** - Laravel 12
